/* ***************************************************** **
   ch03_divide_and_conquer_with_subquery_factoring.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 3.
   Dziel i rządź dzięki użyciu faktoringu podzapytania
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole

/* -----------------------------------------------------
   Przykładowe fragmenty kodu do rozdziału 3.
   ----------------------------------------------------- */

-- Listing 3.1. Podział piw w zależności od zawartości alkoholu

select
   pa.product_id as p_id
 , p.name        as product_name
 , pa.abv
 , ntile(2) over (
      order by pa.abv, pa.product_id
   ) as alc_class
from product_alcohol pa
join products p
   on p.id = pa.product_id
order by pa.abv, pa.product_id;

-- Listing 3.2. Wyświetlanie rocznej wielkości sprzedaży piwa zaliczanego do pierwszej kategorii pod względem zawartości alkoholu

select
   pac.product_id as p_id
 , extract(year from ms.mth) as yr
 , sum(ms.qty) as yr_qty
from (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
) pac
join monthly_sales ms
   on ms.product_id = pac.product_id
where pac.alc_class = 1
group by
   pac.product_id
 , extract(year from ms.mth)
order by p_id, yr;

-- Listing 3.3. Wyświetlenie jedynie tych lat, w których wielkość sprzedaży była większa niż średnia roczna dla tego rodzaju piwa

select
   p_id, yr, yr_qty
 , round(avg_yr) as avg_yr
from (
   select
      pac.product_id as p_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by pac.product_id
      ) as avg_yr
   from (
      select
         pa.product_id
       , ntile(2) over (
            order by pa.abv, pa.product_id
         ) as alc_class
      from product_alcohol pa
   ) pac
   join monthly_sales ms
      on ms.product_id = pac.product_id
   where pac.alc_class = 1
   group by
      pac.product_id
    , extract(year from ms.mth)
)
where yr_qty > avg_yr
order by p_id, yr;

-- Listing 3.4. Modyfikacja kodu listingu 3.3 z wykorzystaniem faktoryzacji podzapytania

with product_alc_class as (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
), class_one_yearly_sales as (
   select
      pac.product_id as p_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by pac.product_id
      ) as avg_yr
   from product_alc_class pac
   join monthly_sales ms
      on ms.product_id = pac.product_id
   where pac.alc_class = 1
   group by
      pac.product_id
    , extract(year from ms.mth)
)
select
   p_id, yr, yr_qty
 , round(avg_yr) as avg_yr
from class_one_yearly_sales
where yr_qty > avg_yr
order by p_id, yr;

-- Listing 3.5. Alternatywna wersja kodu używająca niezależnych nazwanych podzapytań

with product_alc_class as (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
), yearly_sales as (
   select
      ms.product_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by ms.product_id
      ) as avg_yr
   from monthly_sales ms
   group by
      ms.product_id
    , extract(year from ms.mth)
)
select
   pac.product_id as p_id
 , ys.yr
 , ys.yr_qty
 , round(ys.avg_yr) as avg_yr
from product_alc_class pac
join yearly_sales ys
   on ys.product_id = pac.product_id
where pac.alc_class = 1
and ys.yr_qty > ys.avg_yr
order by p_id, yr;

-- Listing 3.6. Przykład wielokrotnego wykonywania podzapytania

with product_alc_class as (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
), yearly_sales as (
   select
      ms.product_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by ms.product_id
      ) as avg_yr
   from monthly_sales ms
   where ms.product_id in (
      select pac.product_id
      from product_alc_class pac
      where pac.alc_class = 1
   )
   group by
      ms.product_id
    , extract(year from ms.mth)
)
select
   pac.product_id as p_id
 , ys.yr
 , ys.yr_qty
 , round(ys.avg_yr) as avg_yr
from product_alc_class pac
join yearly_sales ys
   on ys.product_id = pac.product_id
where ys.yr_qty > ys.avg_yr
order by p_id, yr;

-- Nieudokumentowana sztuczka polegająca na dodaniu podpowiedzi

with product_alc_class as (
   select /*+ materialize */
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
), yearly_sales as (
   select
      ms.product_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by ms.product_id
      ) as avg_yr
   from monthly_sales ms
   where ms.product_id in (
      select pac.product_id
      from product_alc_class pac
      where pac.alc_class = 1
   )
   group by
      ms.product_id
    , extract(year from ms.mth)
)
select
   pac.product_id as p_id
 , ys.yr
 , ys.yr_qty
 , round(ys.avg_yr) as avg_yr
from product_alc_class pac
join yearly_sales ys
   on ys.product_id = pac.product_id
where ys.yr_qty > ys.avg_yr
order by p_id, yr;

-- Klauzula filtrowania (która zawsze przyjmuje wartość true) dla rownum również wymusza na optymalizatorze materializację wyniku

with product_alc_class as (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      ) as alc_class
   from product_alcohol pa
   where rownum >= 1
), yearly_sales as (
   select
      ms.product_id
    , extract(year from ms.mth) as yr
    , sum(ms.qty) as yr_qty
    , avg(sum(ms.qty)) over (
         partition by ms.product_id
      ) as avg_yr
   from monthly_sales ms
   where ms.product_id in (
      select pac.product_id
      from product_alc_class pac
      where pac.alc_class = 1
   )
   group by
      ms.product_id
    , extract(year from ms.mth)
)
select
   pac.product_id as p_id
 , ys.yr
 , ys.yr_qty
 , round(ys.avg_yr) as avg_yr
from product_alc_class pac
join yearly_sales ys
   on ys.product_id = pac.product_id
where ys.yr_qty > ys.avg_yr
order by p_id, yr;

-- Listing 3.7. Podanie listy nazw kolumn zamiast aliasów kolumn

with product_alc_class (
   product_id, alc_class
) as (
   select
      pa.product_id
    , ntile(2) over (
         order by pa.abv, pa.product_id
      )
   from product_alcohol pa
), yearly_sales (
   product_id, yr, yr_qty, avg_yr
) as (
   select
      ms.product_id
    , extract(year from ms.mth)
    , sum(ms.qty)
    , avg(sum(ms.qty)) over (
         partition by ms.product_id
      )
   from monthly_sales ms
   where ms.product_id in (
      select pac.product_id
      from product_alc_class pac
      where pac.alc_class = 1
   )
   group by
      ms.product_id
    , extract(year from ms.mth)
)
select
   pac.product_id as p_id
 , ys.yr
 , ys.yr_qty
 , round(ys.avg_yr) as avg_yr
from product_alc_class pac
join yearly_sales ys
   on ys.product_id = pac.product_id
where ys.yr_qty > ys.avg_yr
order by p_id, yr;

-- Listing 3.8. „Przeciążanie” tabeli za pomocą danych testowych pobieranych w klauzuli with 

with product_alcohol (
   product_id, sales_volume, abv
) as (
   /* Symulacja tabeli product_alcohol. */
   select 4040, 330, 4.5 from dual union all
   select 4160, 500, 7.0 from dual union all
   select 4280, 330, 8.0 from dual union all
   select 5310, 330, 4.0 from dual union all
   select 5430, 330, 8.5 from dual union all
   select 6520, 500, 6.5 from dual union all
   select 6600, 500, 5.0 from dual union all
   select 7790, 500, 4.5 from dual union all
   select 7870, 330, 6.5 from dual union all
   select 7950, 330, 6.0 from dual
)
/* Zapytanie testujące symulowane dane. */
select
   pa.product_id as p_id
 , p.name        as product_name
 , pa.abv
 , ntile(2) over (
      order by pa.abv, pa.product_id
   ) as alc_class
from product_alcohol pa
join products p
   on p.id = pa.product_id
order by pa.abv, pa.product_id;

/* ***************************************************** */
